#
# Copyright (c) Contributors to the Open 3D Engine Project.
# For complete copyright and license terms please see the LICENSE at the root of this distribution.
#
# SPDX-License-Identifier: Apache-2.0 OR MIT
#

include(CMakeParseArguments)

set(AWSNATIVESDK_PACKAGE_NAME AWSNativeSDK)

set(AWS_BASE_PATH ${CMAKE_CURRENT_LIST_DIR}/${AWSNATIVESDK_PACKAGE_NAME})

# Include Path
set(AWSNATIVESDK_INCLUDE_PATH ${AWS_BASE_PATH}/include)

# Determine the lib path and possible bin path
if (LY_MONOLITHIC_GAME)
    set(AWSNATIVESDK_COMPILE_DEFINITIONS 
        AWS_CUSTOM_MEMORY_MANAGEMENT 
        PLATFORM_SUPPORTS_AWS_NATIVE_SDK
    )
    set(AWSNATIVE_SDK_LIB_PATH ${AWS_BASE_PATH}/lib/$<IF:$<CONFIG:Debug>,Debug,Release>)
else()
    set(AWSNATIVESDK_COMPILE_DEFINITIONS 
        AWS_CUSTOM_MEMORY_MANAGEMENT 
        PLATFORM_SUPPORTS_AWS_NATIVE_SDK 
        USE_IMPORT_EXPORT
        USE_WINDOWS_DLL_SEMANTICS
        AWS_CRT_CPP_USE_IMPORT_EXPORT
        AWS_HTTP_USE_IMPORT_EXPORT
        AWS_IO_USE_IMPORT_EXPORT
        AWS_USE_IO_COMPLETION_PORTS
        AWS_COMMON_USE_IMPORT_EXPORT
        AWS_CAL_USE_IMPORT_EXPORT
        AWS_COMPRESSION_USE_IMPORT_EXPORT
        AWS_MQTT_USE_IMPORT_EXPORT
        AWS_MQTT_WITH_WEBSOCKETS
        AWS_AUTH_USE_IMPORT_EXPORT
        AWS_CHECKSUMS_USE_IMPORT_EXPORT
        AWS_EVENT_STREAM_USE_IMPORT_EXPORT
        AWS_S3_USE_IMPORT_EXPORT
    )
    set(AWSNATIVE_SDK_LIB_PATH ${AWS_BASE_PATH}/bin/$<IF:$<CONFIG:Debug>,Debug,Release>)
endif()

# Helper function to define individual AWSNativeSDK Libraries
function(ly_declare_aws_library)
    set(options)
    set(oneValueArgs NAME LIB_FILE)
    set(multiValueArgs BUILD_DEPENDENCIES RUNTIME_DEPENDENCIES)

    cmake_parse_arguments(ly_declare_aws_library "${options}" "${oneValueArgs}" "${multiValueArgs}" ${ARGN})

    set(TARGET_WITH_NAMESPACE "3rdParty::${AWSNATIVESDK_PACKAGE_NAME}::${ly_declare_aws_library_NAME}")
    if (NOT TARGET ${TARGET_WITH_NAMESPACE})
        add_library(${TARGET_WITH_NAMESPACE} INTERFACE IMPORTED GLOBAL)

        ly_target_include_system_directories(TARGET ${TARGET_WITH_NAMESPACE} INTERFACE ${AWSNATIVESDK_INCLUDE_PATH})

        if (ly_declare_aws_library_LIB_FILE)
            target_link_libraries(${TARGET_WITH_NAMESPACE} 
                INTERFACE
                    ${AWSNATIVE_SDK_LIB_PATH}/${CMAKE_STATIC_LIBRARY_PREFIX}${ly_declare_aws_library_LIB_FILE}${CMAKE_STATIC_LIBRARY_SUFFIX}
                    ${ly_declare_aws_library_BUILD_DEPENDENCIES}
            )

            if (NOT LY_MONOLITHIC_GAME)
                ly_add_dependencies(${TARGET_WITH_NAMESPACE} ${AWSNATIVE_SDK_LIB_PATH}/${CMAKE_SHARED_LIBRARY_PREFIX}${ly_declare_aws_library_LIB_FILE}${CMAKE_SHARED_LIBRARY_SUFFIX})

                if (ly_declare_aws_library_RUNTIME_DEPENDENCIES)
                    ly_add_dependencies(${TARGET_WITH_NAMESPACE} ${ly_declare_aws_library_RUNTIME_DEPENDENCIES})
                endif()
            endif()
        elseif (ly_declare_aws_library_BUILD_DEPENDENCIES)
            target_link_libraries(${TARGET_WITH_NAMESPACE} 
                INTERFACE
                    ${ly_declare_aws_library_BUILD_DEPENDENCIES}
            )
        endif()

        target_link_options(${TARGET_WITH_NAMESPACE} INTERFACE ${AWSNATIVESDK_LINK_OPTIONS})

        target_compile_definitions(${TARGET_WITH_NAMESPACE} INTERFACE ${AWSNATIVESDK_COMPILE_DEFINITIONS})

    endif()
endfunction()


#### Core ####
set(CORE_STATIC_LIBS
    ${AWSNATIVE_SDK_LIB_PATH}/aws-crt-cpp.lib
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-s3.lib
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-auth.lib
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-mqtt.lib
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-http.lib
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-event-stream.lib
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-io.lib
    ${AWSNATIVE_SDK_LIB_PATH}/aws-checksums.lib
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-compression.lib
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-cal.lib
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-common.lib
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-sdkutils.lib
    Ncrypt
    Secur32
    Shlwapi
    Userenv
    Wininet
    Winhttp
)

set(CORE_SHARED_LIBS
    ${AWSNATIVE_SDK_LIB_PATH}/aws-crt-cpp.dll
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-s3.dll
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-auth.dll
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-mqtt.dll
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-http.dll
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-event-stream.dll
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-io.dll
    ${AWSNATIVE_SDK_LIB_PATH}/aws-checksums.dll
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-compression.dll
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-cal.dll
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-common.dll
    ${AWSNATIVE_SDK_LIB_PATH}/aws-c-sdkutils.dll
)

if(LY_MONOLITHIC_GAME)
    ly_declare_aws_library(
        NAME 
            Core
        LIB_FILE 
            aws-cpp-sdk-core
        BUILD_DEPENDENCIES
            ${CORE_STATIC_LIBS}
    )
else()
    ly_declare_aws_library(
        NAME 
            Core
        LIB_FILE 
            aws-cpp-sdk-core
        BUILD_DEPENDENCIES
            ${CORE_STATIC_LIBS}
        RUNTIME_DEPENDENCIES
            ${CORE_SHARED_LIBS}
    )
endif()

#### AccessManagement ####
ly_declare_aws_library(
    NAME 
        AccessManagement
    LIB_FILE 
        aws-cpp-sdk-access-management
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)

#### BedrockRuntime ####
ly_declare_aws_library(
    NAME
        BedrockRuntime
    LIB_FILE
        aws-cpp-sdk-bedrock-runtime
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)

#### CognitoIdentity ####
ly_declare_aws_library(
    NAME 
        CognitoIdentity
    LIB_FILE 
        aws-cpp-sdk-cognito-identity
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)

#### CognitoIdp ####
ly_declare_aws_library(
    NAME 
        CognitoIdp
    LIB_FILE 
        aws-cpp-sdk-cognito-idp
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)

#### DeviceFarm ####
ly_declare_aws_library(
    NAME 
        DeviceFarm
    LIB_FILE 
        aws-cpp-sdk-devicefarm
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)

#### DynamoDB ####
ly_declare_aws_library(
    NAME 
        DynamoDB
    LIB_FILE 
        aws-cpp-sdk-dynamodb
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)

#### GameLift ####
ly_declare_aws_library(
    NAME 
        GameLift
    LIB_FILE 
        aws-cpp-sdk-gamelift
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)

#### IdentityManagement ####
ly_declare_aws_library(
    NAME 
        IdentityManagement
    LIB_FILE 
        aws-cpp-sdk-identity-management
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)

#### Kinesis ####
ly_declare_aws_library(
    NAME 
        Kinesis
    LIB_FILE 
        aws-cpp-sdk-kinesis
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)

#### Lambda ####
ly_declare_aws_library(
    NAME 
        Lambda
    LIB_FILE 
        aws-cpp-sdk-lambda
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)

#### MobileAnalytics ####
# MobileAnalytics library no longer exists and should be replaced with AWS PinPoint or other equivalent API

#### Queues ####
ly_declare_aws_library(
    NAME 
        Queues
    LIB_FILE 
        aws-cpp-sdk-queues
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)

#### S3 ####
ly_declare_aws_library(
    NAME 
        S3
    LIB_FILE 
        aws-cpp-sdk-s3
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)

#### SNS ####
ly_declare_aws_library(
    NAME 
        SNS
    LIB_FILE 
        aws-cpp-sdk-sns
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)

#### SQS ####
ly_declare_aws_library(
    NAME 
        SQS
    LIB_FILE 
        aws-cpp-sdk-sqs
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)

#### STS ####
ly_declare_aws_library(
    NAME 
        STS
    LIB_FILE 
        aws-cpp-sdk-sts
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)

#### Transfer ####
ly_declare_aws_library(
    NAME 
        Transfer
    LIB_FILE 
        aws-cpp-sdk-transfer
    BUILD_DEPENDENCIES
        # Even though AWS Core is a sibling module, make sure it's linked in first because this module depends on it.
        3rdParty::AWSNativeSDK::Core 
)


#########
######### Grouping Definitions #########
#########
#### GameLiftClient ####
ly_declare_aws_library(
    NAME 
        GameLiftClient
    BUILD_DEPENDENCIES 
        3rdParty::AWSNativeSDK::Core
        3rdParty::AWSNativeSDK::GameLift
)

#### AWSClientAuth ####
ly_declare_aws_library(
    NAME 
        AWSClientAuth
    BUILD_DEPENDENCIES 
        3rdParty::AWSNativeSDK::Core
        3rdParty::AWSNativeSDK::CognitoIdentity
        3rdParty::AWSNativeSDK::CognitoIdp
        3rdParty::AWSNativeSDK::STS
        3rdParty::AWSNativeSDK::IdentityManagement
)


#### AWSCore ####
ly_declare_aws_library(
    NAME 
        AWSCore
    BUILD_DEPENDENCIES 
        3rdParty::AWSNativeSDK::Core
        3rdParty::AWSNativeSDK::DynamoDB
        3rdParty::AWSNativeSDK::Lambda
        3rdParty::AWSNativeSDK::S3
)
