#
# Copyright (c) Contributors to the Open 3D Engine Project.
# For complete copyright and license terms please see the LICENSE at the root of this distribution.
# 
# SPDX-License-Identifier: Apache-2.0 OR MIT
#
#

set(OPENSSL_O3DE_NAMESPACE "3rdParty::OpenSSL")
if (TARGET ${OPENSSL_O3DE_NAMESPACE})
    return()
endif()

set(SSL_TARGETNAME "OpenSSL::SSL")
set(CRYPTO_TARGETNAME "OpenSSL::Crypto")

# we're trying to be a drop-in replacement for the FindOpenSSL.cmake that is shipped
# with CMake itself, so we set the same variables with the same uppercase for compatibility
# for questions about these variables, see https://cmake.org/cmake/help/latest/module/FindOpenSSL.html
set(OPENSSL_FOUND True)
set(OPENSSL_INCLUDE_DIR ${CMAKE_CURRENT_LIST_DIR}/OpenSSL/include)
# c-only packages can be released containing only Release executables
set(OPENSSL_LIBS_DIR ${CMAKE_CURRENT_LIST_DIR}/OpenSSL/lib)
set(OPENSSL_CRYPTO_LIBRARY ${OPENSSL_LIBS_DIR}/libcrypto${CMAKE_STATIC_LIBRARY_SUFFIX})
set(OPENSSL_CRYPTO_LIBRARIES
    ${OPENSSL_CRYPTO_LIBRARY}
    crypt32.lib ws2_32.lib)
set(OPENSSL_SSL_LIBRARY ${OPENSSL_LIBS_DIR}/libssl${CMAKE_STATIC_LIBRARY_SUFFIX})
set(OPENSSL_SSL_LIBRARIES 
    ${OPENSSL_SSL_LIBRARY}
    ${OPENSSL_CRYPTO_LIBRARIES})
set(OPENSSL_LIBRARIES ${OPENSSL_SSL_LIBRARIES})
set(OPENSSL_VERSION "1.1.1o")

add_library(${CRYPTO_TARGETNAME} STATIC IMPORTED GLOBAL)
set_target_properties(${CRYPTO_TARGETNAME} PROPERTIES IMPORTED_LINK_INTERFACE_LANGUAGES "C")
set_target_properties(${CRYPTO_TARGETNAME} PROPERTIES IMPORTED_LOCATION "${OPENSSL_CRYPTO_LIBRARY}")

# anyone who links to the CRYPTO target also links to its dependencies:
target_link_libraries(${CRYPTO_TARGETNAME} INTERFACE crypt32.lib ws2_32.lib)

add_library(${SSL_TARGETNAME} STATIC IMPORTED GLOBAL)
set_target_properties(${SSL_TARGETNAME} PROPERTIES IMPORTED_LINK_INTERFACE_LANGUAGES "C")
set_target_properties(${SSL_TARGETNAME} PROPERTIES IMPORTED_LOCATION "${OPENSSL_SSL_LIBRARY}")

# anyone who links to the SSL target also links to CRYPTO since SSL depends on CRYPTO:
target_link_libraries(${SSL_TARGETNAME} INTERFACE ${CRYPTO_TARGETNAME})

# cmake < 3.21 and visual studio < 16.10 don't properly implement SYSTEM includes
# so we use O3DEs patched implementation if it is available and fallback to default if not.
# this is futureproof so that when O3DE no longer needs to define this and CMake's system 
# works without fixes, O3DE can erase this implementation and this script will still function.
if (COMMAND ly_target_include_system_directories)
    ly_target_include_system_directories(TARGET ${SSL_TARGETNAME} INTERFACE ${OPENSSL_INCLUDE_DIR})
    ly_target_include_system_directories(TARGET ${CRYPTO_TARGETNAME} INTERFACE ${OPENSSL_INCLUDE_DIR})
else()
    target_include_directories(${SSL_TARGETNAME} SYSTEM INTERFACE ${OPENSSL_INCLUDE_DIR})
    target_include_directories(${CRYPTO_TARGETNAME} SYSTEM INTERFACE ${OPENSSL_INCLUDE_DIR})
endif()

# alias the O3DE name to the official name:
add_library(${OPENSSL_O3DE_NAMESPACE} ALIAS ${SSL_TARGETNAME})

# if we're not in O3DE, it's also extremely helpful to show a message to logs that indicate that this
# library was successfully picked up, as opposed to the system one.
# A good way to know if you're in O3DE or not is that O3DE sets various cache variables before 
# calling find_package, specifically, LY_VERSION_ENGINE_NAME is always set very early:
if (NOT LY_VERSION_ENGINE_NAME)
    message(STATUS "Using O3DE's OpenSSL (${OPENSSL_VERSION}) from ${CMAKE_CURRENT_LIST_DIR}")
endif()
